/*
 * Demo ET-ESP32-RS485 Hardware Board
 * MCU      : ESP32-WROVER
 *          : Arduino Tools Board : ESP32 Wrover Module
 * Software : https://dl.espressif.com/dl/package_esp32_index.json        
 * Reserve  : Boot Config       
 *          : -> IO0(BOOT)
 *          : -> IO2(Don't Card on 3V3V Version)
 *          : -> IO5
 *          : -> IO12(MTDI)
 *          : -> IO15(MTDO)
 * Reserve  : SFLASH        
 *          : -> CMD
 *          : -> CLK
 *          : -> SD0
 *          : -> SD1
 *          : -> SD2
 *          : -> SD3
 * Debug    : Serial0 : USART0(USB)        
 *          : -> RX0(Debug & Download:IO3)
 *          : -> TX0(Debug & Download:IO1)
 * NB-IoT   : Serial1 : SIM7020E(BK-7020 V2)       
 *          : -> RX1(T:IO14)
 *          : -> TX1(R:IO13)
 *          : -> PWRKEY(K:IO33)
 *          : -> SLEEP(S:IO32)
 * RS485    : Serial1 : RS485  
 *          : -> RX2(IO26)
 *          : -> TX2(IO27)
 *          : -> DIR(Direction : IO25 : LOW = Receive, HIGH = Send)
 * I2C#1    : I2C BUS
 *          : -> SCL1(IO22)
 *          : -> SDA1(IO21)
 * RTC      : -> RTC:DS3231/DS3232
 *          : -> PCF8574/A(Relay8)
 *          : -> PCF8574/A(DC-IN8)
 * LED      : LED Status(Active High)
 *          : -> LED0(IO2)
 * Demo     : QC TEST System Hardware Interface         
 */

//=================================================================================================
#include <Wire.h> 
//=================================================================================================
#include "ET_DS3231.h"
//=================================================================================================
#include "ETT_24LC16B.h"
//=================================================================================================
#include "ETT_PCF8574.h"
//=================================================================================================
#include "ETT_PCA9548.h"
//=================================================================================================
#include "DS2482.h"
//=================================================================================================
#include "ETT_ModbusRTU.h"
//=================================================================================================
#include "FS.h"
#include "SD.h"
#include "SPI.h" 
//=================================================================================================

//=================================================================================================
// Start of Default Hardware : ET-ESP32-RS485
//=================================================================================================
// Remap Pin USART -> C:\Users\Admin\Documents\Arduino\hardware\espressif\esp32\cores\esp32\HardwareSerial.cpp
//                    C:\Users\Admin\AppData\Local\Arduino15\packages\esp32\hardware\esp32\1.0.0\cores\esp32\HardwareSerial.cpp
//=================================================================================================
#include <HardwareSerial.h>
//=================================================================================================
#define SerialDebug  Serial                                                                       // USB Serial(Serial0)
//=================================================================================================
#define SerialNBIOT_RX_PIN    14
#define SerialNBIOT_TX_PIN    13
#define SerialNBIOT  Serial1                                                                      // Serial1(IO13=TXD,IO14=RXD)
//=================================================================================================
#define SerialRS485_RX_PIN    26
#define SerialRS485_TX_PIN    27
#define SerialRS485  Serial2                                                                      // Serial2(IO27=TXD,IO26=RXD)
//=================================================================================================
#define RS485_DIRECTION_PIN   25                                                                  // ESP32-WROVER :IO25
#define RS485_RXD_SELECT      LOW
#define RS485_TXD_SELECT      HIGH
//=================================================================================================
#define SIM7020E_PWRKEY_PIN   33                                                                  // ESP32-WROVER :IO33
#define SIM7020E_SLEEP_PIN    32                                                                  // ESP32-WROVER :IO32
#define SIM7020E_PWRKEY_LOW   LOW                                                                 // Start Power-ON
#define SIM7020E_PWRKEY_HIGH  HIGH                                                                // Release Signal
#define SIM7020E_SLEEP_LOW    LOW                                                                 // Pull-Up DTR to Enable Sleep
#define SIM7020E_SLEEP_HIGH   HIGH                                                                // DTR=Low(Wakeup From Sleep)
//=================================================================================================
#define I2C_SCL1_PIN          22                                                                  // ESP32-WROVER : IO22(SCL1)
#define I2C_SDA1_PIN          21                                                                  // ESP32-WROVER : IO21(SDA1)
//=================================================================================================

//=================================================================================================
#define LED_PIN               2                                                                   // ESP-WROVER  : IO2
#define LedON                 1
#define LedOFF                0
//=================================================================================================
#define CS_SD_CARD_PIN        4                                                                   // ESP-WROVER  : IO4
#define SD_CARD_DISABLE       1
#define SD_CARD_ENABLE        0
//=================================================================================================
// End of Default Hardware : ET-ESP32-RS485
//=================================================================================================

//=================================================================================================
// ET_DS3231.h
//=================================================================================================
ET_DS3231 myRTC;
DateTime myTimeNow;
//=================================================================================================
float UTCOffset = +7.0;    // Your timezone relative to UTC (http://en.wikipedia.org/wiki/UTC_offset)
char daysOfTheWeek[8][12] = {"Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday", "Sunday"};
//=================================================================================================
unsigned long lastSecondTime = 0;
//=================================================================================================

//=================================================================================================
ETT_24LC16B myConfig = ETT_24LC16B(0x50);
//=================================================================================================

//=================================================================================================
ETT_PCF8574 master_relay0(PCF8574_ID_DEV0);                                                       // ET-ESP32-RS485  : Output Relay(PCF8574:ID0)
ETT_PCF8574 master_input0(PCF8574A_ID_DEV0);                                                      // ET-ESP32-RS485  : Input OPTO(PCF8574A:ID0) 
ETT_PCF8574 master_relay1(PCF8574_ID_DEV1);                                                       // ET-ESP32-RS485  : Output Relay(PCF8574:ID1)
ETT_PCF8574 master_input1(PCF8574A_ID_DEV1);                                                      // ET-ESP32-RS485  : Input OPTO(PCF8574A:ID1) 
ETT_PCF8574 master_relay2(PCF8574_ID_DEV2);                                                       // ET-ESP32-RS485  : Output Relay(PCF8574:ID2)
ETT_PCF8574 master_input2(PCF8574A_ID_DEV2);                                                      // ET-ESP32-RS485  : Input OPTO(PCF8574A:ID2) 
ETT_PCF8574 master_relay3(PCF8574_ID_DEV3);                                                       // ET-ESP32-RS485  : Output Relay(PCF8574:ID3)
ETT_PCF8574 master_input3(PCF8574A_ID_DEV3);                                                      // ET-ESP32-RS485  : Input OPTO(PCF8574A:ID3)
ETT_PCF8574 master_relay4(PCF8574_ID_DEV4);                                                       // ET-ESP32-RS485  : Output Relay(PCF8574:ID4)
ETT_PCF8574 master_input4(PCF8574A_ID_DEV4);                                                      // ET-ESP32-RS485  : Input OPTO(PCF8574A:ID4)
ETT_PCF8574 master_relay5(PCF8574_ID_DEV5);                                                       // ET-ESP32-RS485  : Output Relay(PCF8574:ID5)
ETT_PCF8574 master_input5(PCF8574A_ID_DEV5);                                                      // ET-ESP32-RS485  : Input OPTO(PCF8574A:ID5)
ETT_PCF8574 master_relay6(PCF8574_ID_DEV6);                                                       // ET-ESP32-RS485  : Output Relay(PCF8574:ID6)
ETT_PCF8574 master_input6(PCF8574A_ID_DEV6);                                                      // ET-ESP32-RS485  : Input OPTO(PCF8574A:ID6) 
ETT_PCF8574 master_relay7(PCF8574_ID_DEV7);                                                       // ET-ESP32-RS485  : Output Relay(PCF8574:ID7)
ETT_PCF8574 master_input7(PCF8574A_ID_DEV7);                                                      // ET-ESP32-RS485  : Input OPTO(PCF8574A:ID7)    
//=================================================================================================
uint8_t state;
unsigned long lastGetDelayTime = 0;
//=================================================================================================

//=================================================================================================
ETT_PCA9548 i2c_select0(PCA9548_ID_DEV0);                                                         // Slave Select Device[0]
ETT_PCF8574 slave_unit0(PCF8574A_ID_DEV7);                                                        // Slave Unit[0] (Relay & OPTO-Input)
ETT_PCA9548 i2c_select1(PCA9548_ID_DEV1);                                                         // Slave Select Device[1]
ETT_PCF8574 slave_unit1(PCF8574A_ID_DEV7);                                                        // Slave Unit[0] (Relay & OPTO-Input)
ETT_PCA9548 i2c_select2(PCA9548_ID_DEV2);                                                         // Slave Select Device[2]
ETT_PCF8574 slave_unit2(PCF8574A_ID_DEV7);                                                        // Slave Unit[0] (Relay & OPTO-Input)
ETT_PCA9548 i2c_select3(PCA9548_ID_DEV3);                                                         // Slave Select Device[3]
ETT_PCF8574 slave_unit3(PCF8574A_ID_DEV7);                                                        // Slave Unit[0] (Relay & OPTO-Input)
ETT_PCA9548 i2c_select4(PCA9548_ID_DEV4);                                                         // Slave Select Device[4]
ETT_PCF8574 slave_unit4(PCF8574A_ID_DEV7);                                                        // Slave Unit[0] (Relay & OPTO-Input)
ETT_PCA9548 i2c_select5(PCA9548_ID_DEV5);                                                         // Slave Select Device[5]
ETT_PCF8574 slave_unit5(PCF8574A_ID_DEV7);                                                        // Slave Unit[0] (Relay & OPTO-Input)
ETT_PCA9548 i2c_select6(PCA9548_ID_DEV6);                                                         // Slave Select Device[6]
ETT_PCF8574 slave_unit6(PCF8574A_ID_DEV7);                                                        // Slave Unit[0] (Relay & OPTO-Input)
ETT_PCA9548 i2c_select7(PCA9548_ID_DEV7);                                                         // Slave Select Device[7]
ETT_PCF8574 slave_unit7(PCF8574A_ID_DEV7);                                                        // Slave Unit[0] (Relay & OPTO-Input)
//=================================================================================================

//=================================================================================================
DS2482 ds_dev0(0);  
//=================================================================================================
byte i;
byte present = 0;
byte type_s;
byte data[12];
byte addr[8];
float celsius, fahrenheit;
//=================================================================================================
unsigned long lastGetI2CSensorTime = 0;
//=================================================================================================

//=================================================================================================
// data array for modbus network sharing
//=================================================================================================
uint16_t au16dataRelay[2];                                                                       // data array for modbus network sharing
uint8_t u8state;                                                                                 // machine state
uint8_t u8query;                                                                                 // pointer to message query
//=================================================================================================

/**
 *  Modbus object declaration
 *  u8id : node id = 0 for master, = 1..247 for slave
 *  u8serno : serial port (use 0 for Serial)
 *  u8txenpin : 0 for RS-232 and USB-FTDI 
 *               or any pin number > 1 for RS-485
 */
Modbus master(0,                                                                                 // node id = 0(master)
              SerialRS485,                                                                       // Serial(2)
              RS485_DIRECTION_PIN);                                                              // RS485 Modbus

/**
 * This is an structe which contains a query to an slave device
 */
modbus_t telegram[2];                                                                            // 2-Modbus Commans
unsigned long u32wait;
//=================================================================================================

int rxByte;

//=================================================================================================
// Start of Function Test SD Card
//=================================================================================================
void listDir(fs::FS &fs, const char * dirname, uint8_t levels)
{
  Serial.printf("Listing directory: %s\n", dirname);

  File root = fs.open(dirname);
  if(!root)
  {
    Serial.println("Failed to open directory");
    return;
  }
  if(!root.isDirectory())
  {
    Serial.println("Not a directory");
    return;
  }

  File file = root.openNextFile();
  while(file)
  {
    if(file.isDirectory())
    {
      Serial.print("  DIR : ");
      Serial.println(file.name());
      if(levels)
      {
        listDir(fs, file.name(), levels -1);
      }
    } 
    else 
    {
      Serial.print("  FILE: ");
      Serial.print(file.name());
      Serial.print("  SIZE: ");
      Serial.println(file.size());
    }
    file = root.openNextFile();
  }
}

void createDir(fs::FS &fs, const char * path)
{
  Serial.printf("Creating Dir: %s\n", path);
  if(fs.mkdir(path))
  {
    Serial.println("Dir created");
  } 
  else 
  {
    Serial.println("mkdir failed");
  }
}

void removeDir(fs::FS &fs, const char * path)
{
  Serial.printf("Removing Dir: %s\n", path);
  if(fs.rmdir(path))
  {
    Serial.println("Dir removed");
  } 
  else 
  {
    Serial.println("rmdir failed");
  }
}

void readFile(fs::FS &fs, const char * path)
{
  Serial.printf("Reading file: %s\n", path);

  File file = fs.open(path);
  if(!file)
  {
    Serial.println("Failed to open file for reading");
    return;
  }

  Serial.print("Read from file: ");
  while(file.available())
  {
    Serial.write(file.read());
  }
}

void writeFile(fs::FS &fs, const char * path, const char * message)
{
  Serial.printf("Writing file: %s\n", path);

  File file = fs.open(path, FILE_WRITE);
  if(!file)
  {
    Serial.println("Failed to open file for writing");
    return;
  }
  if(file.print(message))
  {
    Serial.println("File written");
  } 
  else 
  {
    Serial.println("Write failed");
  }
}

void appendFile(fs::FS &fs, const char * path, const char * message)
{
  Serial.printf("Appending to file: %s\n", path);

  File file = fs.open(path, FILE_APPEND);
  if(!file)
  {
    Serial.println("Failed to open file for appending");
    return;
  }
  if(file.print(message))
  {
    Serial.println("Message appended");
  } 
  else 
  {
    Serial.println("Append failed");
  }
}

void renameFile(fs::FS &fs, const char * path1, const char * path2)
{
  Serial.printf("Renaming file %s to %s\n", path1, path2);
  if(fs.rename(path1, path2)) 
  {
    Serial.println("File renamed");
  } 
  else 
  {
    Serial.println("Rename failed");
  }
}

void deleteFile(fs::FS &fs, const char * path)
{
  Serial.printf("Deleting file: %s\n", path);
  if(fs.remove(path))
  {
    Serial.println("File deleted");
  } 
  else 
  {
    Serial.println("Delete failed");
  }
}

void testFileIO(fs::FS &fs, const char * path)
{
  File file = fs.open(path);
  static uint8_t buf[512];
  size_t len = 0;
  uint32_t start = millis();
  uint32_t end = start;
  if(file)
  {
    len = file.size();
    size_t flen = len;
    start = millis();
    while(len)
    {
      size_t toRead = len;
      if(toRead > 512)
      {
        toRead = 512;
      }
      file.read(buf, toRead);
      len -= toRead;
    }
    end = millis() - start;
    Serial.printf("%u bytes read for %u ms\n", flen, end);
    file.close();
  } 
  else 
  {
    Serial.println("Failed to open file for reading");
  }

  file = fs.open(path, FILE_WRITE);
  if(!file)
  {
    Serial.println("Failed to open file for writing");
    return;
  }

  size_t i;
  start = millis();
  for(i=0; i<2048; i++)
  {
    file.write(buf, 512);
  }
  end = millis() - start;
  Serial.printf("%u bytes written for %u ms\n", 2048 * 512, end);
  file.close();
}
//=================================================================================================
// End of Function Test SD Card
//=================================================================================================

void setup() 
{
  //===============================================================================================
  // Start of Initial Default Hardware : ET-ESP32-RS485
  //===============================================================================================
  pinMode(LED_PIN, OUTPUT);
  digitalWrite(LED_PIN, LedOFF);
  //===============================================================================================
  pinMode(CS_SD_CARD_PIN, OUTPUT);
  digitalWrite(CS_SD_CARD_PIN, SD_CARD_DISABLE);
  //===============================================================================================
  Wire.begin(I2C_SDA1_PIN,I2C_SCL1_PIN);                                                      
  //===============================================================================================
  pinMode(RS485_DIRECTION_PIN, OUTPUT);                                                          // RS485 Direction
  digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
  //===============================================================================================
  SerialDebug.begin(115200);
  while(!SerialDebug);
  //===============================================================================================
  SerialNBIOT.begin(115200, SERIAL_8N1, SerialNBIOT_RX_PIN, SerialNBIOT_TX_PIN);
  while(!SerialNBIOT);
  //===============================================================================================
  SerialRS485.begin(9600, SERIAL_8N1, SerialRS485_RX_PIN, SerialRS485_TX_PIN);
  while(!SerialRS485);
  //===============================================================================================
  // End of Initial Default Hardware : ESP32
  //===============================================================================================
  
  //===============================================================================================
  i2c_select0.begin();                                                                           // Initial ET-I2C Longlngth Repeater : Slave[0]
  i2c_select0.disableBus();
  //===============================================================================================
  
  //===============================================================================================
  SerialDebug.println();
  SerialDebug.println("ET-ESP32-RS485...Hardware Test");
  print_menu_test();  
  //===============================================================================================
}

void loop() 
{
  //===============================================================================================
  if(SerialDebug.available() > 0)
  {
    rxByte = SerialDebug.read();
    
    if(rxByte == '1') 
    {
      //===========================================================================================
      SerialDebug.println();
      SerialDebug.println("==============================================================");
      SerialDebug.println("1. Test Relay & OPTO Input");
      SerialDebug.println("==============================================================");
      //===========================================================================================
      test_internal_io();
      //===========================================================================================
    }
    
    else if(rxByte == '2') 
    {
      //===========================================================================================
      SerialDebug.println();
      SerialDebug.println("==============================================================");
      SerialDebug.println("2. Test RTC DS3231");
      SerialDebug.println("==============================================================");
      //===========================================================================================
      test_rtc();
      //===========================================================================================
    }

    else if(rxByte == '3') 
    {
      //===========================================================================================
      SerialDebug.println();
      SerialDebug.println("==============================================================");
      SerialDebug.println("3. Test EEPROM 24LC16B");
      SerialDebug.println("==============================================================");
      //===========================================================================================
      test_24lc16b();
      //===========================================================================================
    }
    
    else if(rxByte == '4') 
    {
      //===========================================================================================
      SerialDebug.println();
      SerialDebug.println("==============================================================");
      SerialDebug.println("4. Test 1-Wire DS1820");
      SerialDebug.println("==============================================================");
      //===========================================================================================
      test_1wire();
      //===========================================================================================
    }

    else if(rxByte == '5') 
    {
      //===========================================================================================
      SerialDebug.println();
      SerialDebug.println("==============================================================");
      SerialDebug.println("5. Test RS422/RS485 4-Wire");
      SerialDebug.println("==============================================================");
      //===========================================================================================
      test_rs422();
      //===========================================================================================
    }
    
    else if(rxByte == '6') 
    {
      //===========================================================================================
      SerialDebug.println();
      SerialDebug.println("==============================================================");
      SerialDebug.println("6. Test RS485 2 Wire(Modbus RTU : Modbus Relay2)");
      SerialDebug.println("==============================================================");
      //===========================================================================================
      test_rs485_modbus();
      //===========================================================================================
    }

    else if(rxByte == '7') 
    {
      //===========================================================================================
      SerialDebug.println();
      SerialDebug.println("==============================================================");
      SerialDebug.println("7. Test SD Card");
      SerialDebug.println("==============================================================");
      //===========================================================================================
      test_sd_card();
      //===========================================================================================
    }

    else if(rxByte == '8') 
    {
      //===========================================================================================
      SerialDebug.println();
      SerialDebug.println("==============================================================");
      SerialDebug.println("8. Test I2C Longlength:PCA9548=ID0,PCF8574A=ID7");
      SerialDebug.println("==============================================================");
      //===========================================================================================
      test_i2c_longlength();
      //===========================================================================================
    }
    
    else if(rxByte == 0x0D)             
    {
      print_menu_test();  
    }
  }   
  //===============================================================================================
}


//=================================================================================================
// Start of Print Menu Test
//=================================================================================================
void print_menu_test(void)
{
  //===============================================================================================
  // Start of Menu Test
  //===============================================================================================
  SerialDebug.println();
  SerialDebug.println("==============================================================");
  SerialDebug.println("QC Test : ET-ESP32 RS485 Hardware System");
  SerialDebug.println("==============================================================");
  SerialDebug.println("1. Test Relay & OPTO Input");
  SerialDebug.println("2. Test RTC DS3231");
  SerialDebug.println("3. Test EEPROM 24LC16B");
  SerialDebug.println("4. Test 1-Wire DS1820");
  SerialDebug.println("5. Test RS422/RS485 4-Wire");
  SerialDebug.println("6. Test RS485 2 Wire(Modbus RTU : Modbus Relay2)");
  SerialDebug.println("7. Test SD Card");
  SerialDebug.println("8. Test I2C Longlength:PCA9548=ID0,PCF8574A=ID7");
  SerialDebug.println("==============================================================");
  SerialDebug.println("Select [1..8] For Test");
  SerialDebug.println("==============================================================");
  //===============================================================================================
  // End of Menu Test
  //===============================================================================================
}
//=================================================================================================
// End of Print Menu Test
//=================================================================================================


//=================================================================================================
// Start of Test Relay & OPTO Input
//=================================================================================================
void test_internal_io(void)
{
  //===============================================================================================
  SerialDebug.println("Setup Jumper PCF8574 ID[0..7] & PCF8574A ID[0..7] For Test");
  SerialDebug.println("Active OPTO-Input[0..7] -> Control Output Relay[0..7]");
  //===============================================================================================
  master_relay0.begin(0xFF);
  master_relay1.begin(0xFF);
  master_relay2.begin(0xFF);
  master_relay3.begin(0xFF);
  master_relay4.begin(0xFF);
  master_relay5.begin(0xFF);
  master_relay6.begin(0xFF);
  master_relay7.begin(0xFF);
  //===============================================================================================
  master_input0.begin(0xFF);
  master_input1.begin(0xFF);
  master_input2.begin(0xFF);
  master_input3.begin(0xFF);
  master_input4.begin(0xFF);
  master_input5.begin(0xFF);
  master_input6.begin(0xFF);
  master_input7.begin(0xFF);
  //===============================================================================================
  while(1)
  {
    if(master_relay0.valueOut() != master_input0.readPort())
    {
      master_relay0.writePort(master_input0.readPort());
    }
    
    if(master_relay1.valueOut() != master_input1.readPort())
    {
      master_relay1.writePort(master_input1.readPort());
    }
    
    if(master_relay2.valueOut() != master_input2.readPort())
    {
      master_relay2.writePort(master_input2.readPort());
    }
   
    if(master_relay3.valueOut() != master_input3.readPort())
    {
      master_relay3.writePort(master_input3.readPort());
    }
    
    if(master_relay4.valueOut() != master_input4.readPort())
    {
      master_relay4.writePort(master_input4.readPort());
    }
    
    if(master_relay5.valueOut() != master_input5.readPort())
    {
      master_relay5.writePort(master_input5.readPort());
    }
    
    if(master_relay6.valueOut() != master_input6.readPort())
    {
      master_relay6.writePort(master_input6.readPort());
    }
    
    if(master_relay7.valueOut() != master_input7.readPort())
    {
      master_relay7.writePort(master_input7.readPort());
    }
  }
  //===============================================================================================
}
//=================================================================================================
// End of Test Relay & OPTO Input
//=================================================================================================

//=================================================================================================
// Start of Test RTC:DS3231
//=================================================================================================
void test_rtc(void)
{
  //===============================================================================================
  SerialDebug.println();
  SerialDebug.println("ET-ESP32-RS485...Demo RTC:DS3231 Interface");
  //===============================================================================================
  SerialDebug.println("Initialize DS3231");
  //===============================================================================================
  myRTC.begin();
  //===============================================================================================
  if(myRTC.lostPower()) 
  {
    SerialDebug.println("RTC lost power, lets set the time!");
    myRTC.adjust(DateTime(F(__DATE__), F(__TIME__)));                                            // Setup RTC from date & time this sketch was compiled
  }
  //===============================================================================================
  myRTC.armAlarm1(false);
  myRTC.clearAlarm1();
  //===============================================================================================
  myRTC.armAlarm2(false);
  myRTC.clearAlarm2();
  //===============================================================================================
  
  //===============================================================================================
  myRTC.setAlarm1(0, 0, 0, 0, DS3231_EVERY_SECOND);                                              // Alarm Every Second
  //===============================================================================================
  //===============================================================================================
  SerialDebug.println("Initial RTC:DS3231....Complete");
  //===============================================================================================

  while(1)
  {
    //=============================================================================================
    if(myRTC.isAlarm1(false))
    {
      //===========================================================================================
      myTimeNow = myRTC.now();
      //===========================================================================================
      SerialDebug.print("RTC Time : ");
      SerialDebug.print(daysOfTheWeek[myTimeNow.dayOfTheWeek()]);
      SerialDebug.print(',');
      SerialDebug.print(myTimeNow.day());
      SerialDebug.print('/');
      SerialDebug.print(myTimeNow.month());
      SerialDebug.print('/');
      SerialDebug.print(myTimeNow.year());
      SerialDebug.print(" ");
      SerialDebug.print(myTimeNow.hour());
      SerialDebug.print(':');
      SerialDebug.print(myTimeNow.minute());
      SerialDebug.print(':');
      SerialDebug.print(myTimeNow.second());
      SerialDebug.println();
      //===========================================================================================
                          
      //===========================================================================================
      myRTC.clearAlarm1();
      //===========================================================================================
    }
    //=============================================================================================
    delay(500);
    //=============================================================================================
  }
}
//=================================================================================================
// End of Test RTC:DS3231
//=================================================================================================

//=================================================================================================
// End of Test EEPROM:24LC16B
//=================================================================================================
void test_24lc16b(void)
{
  //===============================================================================================
  SerialDebug.println();
  SerialDebug.println("ET-ESP32-RS485...Demo I2C EEPROM 24LC16B Interface");
  //===============================================================================================
  
  //=============================================================================================== 
  SerialDebug.println();
  SerialDebug.println("Start 24LC16B Witre");
  //===============================================================================================
  char eep_config[] = "hello...this is data from the eeprom 24LC16B";
  myConfig.write(0, (byte *)eep_config, sizeof(eep_config));                                     // Write Address 0...
  //===============================================================================================
  SerialDebug.println(eep_config);
  SerialDebug.println("Write Complete");
  SerialDebug.println();
  //===============================================================================================
  SerialDebug.println("Start 24LC16B Read");
  //===============================================================================================
  int addr=0;                                                                                    // first address
  //===============================================================================================
  byte b =  myConfig.read_byte(addr);                                                            // access the first address from the memory
  //===============================================================================================
  while(b!=0)
  {
    SerialDebug.print((char)b);                                                                  // print content to serial port
    addr++;                                                                                      // increase address
   b = myConfig.read_byte(addr);                                                                 // access an address from the memory
  }
  SerialDebug.println();
  SerialDebug.println("Read Complete");
  //===============================================================================================
  while(1);
  //===============================================================================================
}
//=================================================================================================
// End of Test EEPROM:24LC16B
//=================================================================================================

//=================================================================================================
// Start of Test 1 Wire Bus
//=================================================================================================
void test_1wire(void)
{
  //===============================================================================================
  SerialDebug.println();
  SerialDebug.println("ET-ESP32-RS485...Demo 1-Wire Interface");
  //===============================================================================================
  
  //===============================================================================================
  SerialDebug.println();
  SerialDebug.println("DS2482 I2C() Test");
  //===============================================================================================
  //===============================================================================================
  SerialDebug.println("Initial DS2482...Complete");
  SerialDebug.println();
  //===============================================================================================
  lastGetI2CSensorTime = millis();
  //===============================================================================================
  while(1)
  {
    //=============================================================================================
    do
    {
      //===========================================================================================
      // Start of Device[0] Test
      //===========================================================================================
      if(ds_dev0.wireSearch(addr))
      {
        present = ds_dev0.wireReset(); 
        SerialDebug.print("Found Device(");
        SerialDebug.print(present, HEX);
        //
        //=========================================================================================
        // the first ROM byte indicates which chip
        //=========================================================================================
        switch(addr[0]) 
        {
          case 0x10:
            SerialDebug.print(":(DS18S20)");  // or old DS1820
            type_s = 1;
          break;
    
          case 0x28:
            SerialDebug.print(":(DS18B20)");
            type_s = 0;
          break;
    
          case 0x22:
            SerialDebug.print(":(DS1822)");
            type_s = 0;
          break;
    
          default:
            SerialDebug.print(":Unknow)");
          return;
        } 
        //===========================================================================================
        SerialDebug.print(" ROM =");
      
        for( i = 0; i < 8; i++) 
        {
          SerialDebug.write(' ');
          if(addr[i] < 16) SerialDebug.print("0");
          SerialDebug.print(addr[i], HEX);
        }
        //===========================================================================================
        ds_dev0.wireReset();
        ds_dev0.wireSelect(addr);
        ds_dev0.wireWriteByte(0x44);  // start conversion, use ds.write(0x44,1) with parasite power on at the end
        //===========================================================================================
        delay(1000);     // maybe 750ms is enough, maybe not
        //===========================================================================================
        // we might do a ds.depower() here, but the reset will take care of it.
        //===========================================================================================
        present = ds_dev0.wireReset(); //ds.reset(); 
        ds_dev0.wireSelect(addr);
        ds_dev0.wireWriteByte(0xBE);         // Read Scratchpad
        //===========================================================================================
        //Serial.print("Data(Device:");
        //Serial.print(present, HEX);
        //Serial.print(") = ");
        SerialDebug.print(" Data :");
        for ( i = 0; i < 9; i++) 
        {           // we need 9 bytes
          SerialDebug.print(" ");
          data[i] = ds_dev0.wireReadByte();
          if(data[i] < 16) SerialDebug.print("0");
          SerialDebug.print(data[i], HEX);
        }
        //Serial.print(" CRC = ");
        SerialDebug.print(":CRC(");
        if(ds_dev0.crc8(data, 8) < 16) SerialDebug.print("0");
        SerialDebug.print(ds_dev0.crc8(data, 8), HEX);
        SerialDebug.print(")");
        //===========================================================================================

        //===========================================================================================
        // Convert the data to actual temperature
        // because the result is a 16 bit signed integer, it should
        // be stored to an "int16_t" type, which is always 16 bits
        // even when compiled on a 32 bit processor.
        //===========================================================================================
        int16_t raw = (data[1] << 8) | data[0];
        //===========================================================================================
        if(type_s) 
        {
          raw = raw << 3; // 9 bit resolution default
          if (data[7] == 0x10) 
          {
            // "count remain" gives full 12 bit resolution
            raw = (raw & 0xFFF0) + 12 - data[6];
          }
        } 
        else 
        {
          //=========================================================================================
          byte cfg = (data[4] & 0x60);
          //=========================================================================================
          // at lower res, the low bits are undefined, so let's zero them
          if (cfg == 0x00) raw = raw & ~7;                // 9 bit resolution, 93.75 ms
          else if (cfg == 0x20) raw = raw & ~3;           // 10 bit res, 187.5 ms
          else if (cfg == 0x40) raw = raw & ~1;           // 11 bit res, 375 ms
          //// default is 12 bit resolution, 750 ms conversion time
          //=========================================================================================
        }
        celsius = (float)raw / 16.0;
        fahrenheit = celsius * 1.8 + 32.0;
        SerialDebug.print("  Temperature = ");
        SerialDebug.print(celsius);
        SerialDebug.print(" Celsius, ");
        SerialDebug.print(fahrenheit);
        SerialDebug.println(" Fahrenheit");
        //===========================================================================================
      }
      else
      {
        ds_dev0.wireResetSearch();
        break;
      }
      //=============================================================================================
      // End of Device[0] Test
      //=============================================================================================
    }
    //===============================================================================================
    while(1);
    //===============================================================================================
    delay(2000);
    //===============================================================================================
  }
}
//=================================================================================================
// End of Test 1 Wire Bus
//=================================================================================================

//=================================================================================================
// Start of Test RS422/RS485 4-Wire : Close Loop
//=================================================================================================
void test_rs422(void)
{
  //===============================================================================================
  digitalWrite(RS485_DIRECTION_PIN, RS485_TXD_SELECT);
  SerialRS485.flush();
  //===============================================================================================
  //===============================================================================================
  SerialDebug.println();
  SerialDebug.println("ET-ESP32-RS485...RS422 Close looop Test");
  SerialDebug.println("Setup RS422/485 Jumper");
  SerialDebug.println(" -> FULL/HALF   = FULL");
  SerialDebug.println(" -> 422/485     = 422");
  SerialDebug.println(" -> 4WIRE/2WIRE = 2WIRE(close Loop)");
  SerialDebug.println(" -> RZ(ENA/DIS) = ENA");
  SerialDebug.println(" -> RH(ENA/DIS) = ENA");
  SerialDebug.println(" -> RL(ENA/DIS) = ENA");
  SerialDebug.println();
  SerialDebug.print(">");
  //===============================================================================================
  
  while(1)
  {
    //=============================================================================================
    // Receive Serial0 -> Serial1
    //=============================================================================================
    // Receive Serial1 -> Serial0
    //=============================================================================================
    if(SerialDebug.available() > 0)
    {
      rxByte = SerialDebug.read();
      SerialRS485.write(rxByte);
    }
    //=============================================================================================
  
    //=============================================================================================
    if(SerialRS485.available() > 0)
    {
      rxByte = SerialRS485.read();
      SerialDebug.write(rxByte);
    }
    //=============================================================================================
  }
}
//=================================================================================================
// End of Test RS422/RS485 4-Wire : Close Loop
//=================================================================================================

//=================================================================================================
// Start of Test RS485 Modbus RTU
//=================================================================================================
void test_rs485_modbus(void)
{
  //===============================================================================================
  SerialDebug.println();
  SerialDebug.println("ET-ESP32-RS485...Test RS485 Modbus");
  SerialDebug.println("Interface Modbus RTU Relay2");
  SerialDebug.println("Setup RS422/485 Jumper");
  SerialDebug.println(" -> FULL/HALF   = HALF");
  SerialDebug.println(" -> 422/485     = 485");
  SerialDebug.println(" -> 4WIRE/2WIRE = 2WIRE");
  SerialDebug.println(" -> RZ(ENA/DIS) = ENA");
  SerialDebug.println(" -> RH(ENA/DIS) = ENA");
  SerialDebug.println(" -> RL(ENA/DIS) = ENA");
  //===============================================================================================

  //===============================================================================================
  // Mosbus RTU Command
  //=============================================================================================== 
  //au16dataRelay[0] = 0x0300;                                                                      // Toggle Output Command
  au16dataRelay[0] = 0x0400;                                                                      // Latch Output Command
  //au16dataRelay[0] = 0x0500;                                                                      // Momentary Output Command
  //au16dataRelay[0] = 0x0600;                                                                      // Delaye Output Command
  //===============================================================================================
  // telegram[0]: Toggle Relay1 device[1]
  //            : Modbus Relay RS485(Modbus RTU)
  //            : Device ID[1]
  // Device Addr,Function Code(06),MSB Reg Addr,LSB Reg Addr,MSB Value,LSB Value,CRC16
  //===============================================================================================
  telegram[0].u8id = 1;                                                                           // slave address = 1
  telegram[0].u8fct = 6;                                                                          // function code (toggle relay1)
  telegram[0].u16RegAdd = 0x0001;                                                                 // start address in slave
  telegram[0].u16CoilsNo = 1;                                                                     // number of elements (coils or registers) to read
  telegram[0].au16reg = au16dataRelay;                                                            // pointer to a memory array in the Arduino
  //===============================================================================================
  // telegram[1]: Toggle Relay2 device[1]
  //            : Modbus Relay RS485(Modbus RTU)
  //            : Device ID[1]
  // Device Addr,Function Code(06),MSB Reg Addr,LSB Reg Addr,MSB Value,LSB Value,CRC16
  //===============================================================================================
  telegram[1].u8id = 1;                                                                          // slave address = 1
  telegram[1].u8fct = 6;                                                                         // function code (this one is registers read)
  telegram[1].u16RegAdd = 0x0002;                                                                // start address in slave
  telegram[1].u16CoilsNo = 1;                                                                    // number of elements (coils or registers) to read
  telegram[1].au16reg = au16dataRelay;                                                           // pointer to a memory array in the Arduino
  //===============================================================================================
  master.begin(SerialRS485);                                                                     // Mosbus Interface
  master.setTimeOut(2000);                                                                       // if there is no answer in 2000 ms, roll over
  u32wait = millis() + 500;
  u8state = u8query = 0; 
  //===============================================================================================

  //===============================================================================================
  while(1)
  {
    //=============================================================================================
    switch( u8state ) 
    {
      case 0: 
        if (millis() > u32wait) u8state++;        // wait state
      break;
    
      case 1: 
        master.query(telegram[u8query]);          // send query (only once)
        u8state++;
        u8query++;
        if(u8query > 2) u8query = 0;              // 0,1,...,0
      break;

    
      case 2:
        master.poll();                            // check incoming messages
        if(master.getState() == COM_IDLE) 
        {
          u8state = 0;
          u32wait = millis() + 500; 
        } 
      break;
    }
    //=============================================================================================
  }  
  //===============================================================================================
}
//=================================================================================================
// End of Test RS485 Modbus RTU
//=================================================================================================

//=================================================================================================
// Start of Test SD Card
//=================================================================================================
void test_sd_card(void)
{
  //===============================================================================================
  SerialDebug.println();
  SerialDebug.println("ET-ESP32-RS485...Demo Interface SD Card");
  //===============================================================================================

  //===============================================================================================
  SerialDebug.println();
  SerialDebug.println("SD Card Test...Start");
  //===============================================================================================
  if(!SD.begin())
  {
    //=============================================================================================
    SerialDebug.println("Card Mount Failed");
    //=============================================================================================
    return;
  }
  uint8_t cardType = SD.cardType();

  if(cardType == CARD_NONE)
  {
    //=============================================================================================
    SerialDebug.println("No SD card attached");
    //=============================================================================================
    return;
  }
  //===============================================================================================
  SerialDebug.print("SD Card Type: ");
  //===============================================================================================
  if(cardType == CARD_MMC)
  {
    //=============================================================================================
    SerialDebug.println("MMC");
    //=============================================================================================
  } 
  else if(cardType == CARD_SD)
  {
    //=============================================================================================
    SerialDebug.println("SDSC");
    //=============================================================================================
  } 
  else if(cardType == CARD_SDHC)
  {
    //=============================================================================================
    SerialDebug.println("SDHC");
    //=============================================================================================
  } 
  else 
  {
    //=============================================================================================
    SerialDebug.println("UNKNOWN");
    //=============================================================================================
  }

  uint64_t cardSize = SD.cardSize() / (1024 * 1024);
  //===============================================================================================
  SerialDebug.printf("SD Card Size: %lluMB\n", cardSize);
  //===============================================================================================
  listDir(SD, "/", 0);
  createDir(SD, "/mydir");
  listDir(SD, "/", 0);
  removeDir(SD, "/mydir");
  listDir(SD, "/", 2);
  writeFile(SD, "/hello.txt", "Hello ");
  appendFile(SD, "/hello.txt", "World!\n");
  readFile(SD, "/hello.txt");
  deleteFile(SD, "/foo.txt");
  renameFile(SD, "/hello.txt", "/foo.txt");
  readFile(SD, "/foo.txt");
  testFileIO(SD, "/test.txt");
  //===============================================================================================
  SerialDebug.println("Test SD Card...Complete");
  //===============================================================================================
  
  while(1);  
}
//=================================================================================================
// End of Test SD Card
//=================================================================================================

//=================================================================================================
// Start of Test I2C Longlength
//=================================================================================================
void test_i2c_longlength(void)
{
  //===============================================================================================
  SerialDebug.println("Setup Jumper PCA9548A ID[0..7] & PCF8574A ID[7] For Test");
  SerialDebug.println("Active OPTO-Input[0..1] -> Control Output Relay[0..1]");
  //===============================================================================================
  //===============================================================================================
  i2c_select0.enableBus();                                          // Enable BUS
  slave_unit0.begin(0xFF);                                          // Initial PCF8574 Control Relay[0,1]
  slave_unit0.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);            // Control Relay[0]
  slave_unit0.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);            // Control Relay[1]
  //===============================================================================================
  state = 0;
  //===============================================================================================

  while(1)
  {
    //============================================================================================= 
    i2c_select0.enableBus();
    if(slave_unit0.readPin(SLAVE_RELAY_OUT0_PIN) != slave_unit0.readPin(SLAVE_OPTO_IN0_PIN))
    {
      slave_unit0.writePin(SLAVE_RELAY_OUT0_PIN, slave_unit0.readPin(SLAVE_OPTO_IN0_PIN));
    }
    if(slave_unit0.readPin(SLAVE_RELAY_OUT1_PIN) != slave_unit0.readPin(SLAVE_OPTO_IN1_PIN))
    {
      slave_unit0.writePin(SLAVE_RELAY_OUT1_PIN, slave_unit0.readPin(SLAVE_OPTO_IN1_PIN));
    } 
    i2c_select0.disableBus();
    //============================================================================================= 

    //============================================================================================= 
    i2c_select1.enableBus();
    if(slave_unit1.readPin(SLAVE_RELAY_OUT0_PIN) != slave_unit1.readPin(SLAVE_OPTO_IN0_PIN))
    {
      slave_unit1.writePin(SLAVE_RELAY_OUT0_PIN, slave_unit1.readPin(SLAVE_OPTO_IN0_PIN));
    }
    if(slave_unit1.readPin(SLAVE_RELAY_OUT1_PIN) != slave_unit1.readPin(SLAVE_OPTO_IN1_PIN))
    {
      slave_unit1.writePin(SLAVE_RELAY_OUT1_PIN, slave_unit1.readPin(SLAVE_OPTO_IN1_PIN));
    } 
    i2c_select1.disableBus();
    //============================================================================================= 

    //============================================================================================= 
    i2c_select2.enableBus();
    if(slave_unit2.readPin(SLAVE_RELAY_OUT0_PIN) != slave_unit2.readPin(SLAVE_OPTO_IN0_PIN))
    {
      slave_unit2.writePin(SLAVE_RELAY_OUT0_PIN, slave_unit2.readPin(SLAVE_OPTO_IN0_PIN));
    }
    if(slave_unit2.readPin(SLAVE_RELAY_OUT1_PIN) != slave_unit2.readPin(SLAVE_OPTO_IN1_PIN))
    {
      slave_unit2.writePin(SLAVE_RELAY_OUT1_PIN, slave_unit2.readPin(SLAVE_OPTO_IN1_PIN));
    } 
    i2c_select2.disableBus();
    //============================================================================================= 

    //============================================================================================= 
    i2c_select3.enableBus();
    if(slave_unit3.readPin(SLAVE_RELAY_OUT0_PIN) != slave_unit3.readPin(SLAVE_OPTO_IN0_PIN))
    {
      slave_unit3.writePin(SLAVE_RELAY_OUT0_PIN, slave_unit3.readPin(SLAVE_OPTO_IN0_PIN));
    }
    if(slave_unit3.readPin(SLAVE_RELAY_OUT1_PIN) != slave_unit3.readPin(SLAVE_OPTO_IN1_PIN))
    {
      slave_unit3.writePin(SLAVE_RELAY_OUT1_PIN, slave_unit3.readPin(SLAVE_OPTO_IN1_PIN));
    } 
    i2c_select3.disableBus();
    //============================================================================================= 

    //============================================================================================= 
    i2c_select4.enableBus();
    if(slave_unit4.readPin(SLAVE_RELAY_OUT0_PIN) != slave_unit4.readPin(SLAVE_OPTO_IN0_PIN))
    {
      slave_unit4.writePin(SLAVE_RELAY_OUT0_PIN, slave_unit4.readPin(SLAVE_OPTO_IN0_PIN));
    }
    if(slave_unit4.readPin(SLAVE_RELAY_OUT1_PIN) != slave_unit4.readPin(SLAVE_OPTO_IN1_PIN))
    {
      slave_unit4.writePin(SLAVE_RELAY_OUT1_PIN, slave_unit4.readPin(SLAVE_OPTO_IN1_PIN));
    } 
    i2c_select4.disableBus();
    //============================================================================================= 

    //============================================================================================= 
    i2c_select5.enableBus();
    if(slave_unit5.readPin(SLAVE_RELAY_OUT0_PIN) != slave_unit5.readPin(SLAVE_OPTO_IN0_PIN))
    {
      slave_unit5.writePin(SLAVE_RELAY_OUT0_PIN, slave_unit5.readPin(SLAVE_OPTO_IN0_PIN));
    }
    if(slave_unit5.readPin(SLAVE_RELAY_OUT1_PIN) != slave_unit5.readPin(SLAVE_OPTO_IN1_PIN))
    {
      slave_unit5.writePin(SLAVE_RELAY_OUT1_PIN, slave_unit5.readPin(SLAVE_OPTO_IN1_PIN));
    } 
    i2c_select5.disableBus();
    //============================================================================================= 

    //============================================================================================= 
    i2c_select6.enableBus();
    if(slave_unit6.readPin(SLAVE_RELAY_OUT0_PIN) != slave_unit6.readPin(SLAVE_OPTO_IN0_PIN))
    {
      slave_unit6.writePin(SLAVE_RELAY_OUT0_PIN, slave_unit6.readPin(SLAVE_OPTO_IN0_PIN));
    }
    if(slave_unit6.readPin(SLAVE_RELAY_OUT1_PIN) != slave_unit6.readPin(SLAVE_OPTO_IN1_PIN))
    {
      slave_unit6.writePin(SLAVE_RELAY_OUT1_PIN, slave_unit6.readPin(SLAVE_OPTO_IN1_PIN));
    } 
    i2c_select6.disableBus();
    //============================================================================================= 

    //============================================================================================= 
    i2c_select7.enableBus();
    if(slave_unit7.readPin(SLAVE_RELAY_OUT0_PIN) != slave_unit7.readPin(SLAVE_OPTO_IN0_PIN))
    {
      slave_unit7.writePin(SLAVE_RELAY_OUT0_PIN, slave_unit7.readPin(SLAVE_OPTO_IN0_PIN));
    }
    if(slave_unit7.readPin(SLAVE_RELAY_OUT1_PIN) != slave_unit7.readPin(SLAVE_OPTO_IN1_PIN))
    {
      slave_unit7.writePin(SLAVE_RELAY_OUT1_PIN, slave_unit7.readPin(SLAVE_OPTO_IN1_PIN));
    } 
    i2c_select7.disableBus();
    //============================================================================================= 
  }
  //===============================================================================================  
}
//=================================================================================================
// End of Test Test I2C Longlength
//=================================================================================================

